/* Copyright (c) 2024 Claus-Uwe Hass (DL8LAB)
 *
 * MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/* 
*  "LoRa-Schalter" universal Sketch für Sender und Empfänger.
*  Mit 2 Tasten, 2 LED und einem Relais. 
*  Verwendet wird ein Arduino Pro Mini mit 3,3V und 8MHz, 
*  sowie ein EBYTE E32-400M20S LoRa Modul.
*  Die LoRa-Parameter sind einstellbar.
*  Ein gesendetets Signal wird quittiert, wenn es richtig empfangen wurde.
*  Mit Stausabfrage an den Empfänger beim Einschalten.
*  Hardware-Codiermöglichkeit für 16 verschiedene Systeme, default = 0
*  Hardware-Codierung für die Wiederherstellung des letzten
*  Schaltzustands nach Wiedereinschalten bzw. Spannungsverlust beim Empfänger.
*  11.08.2024
*
*  Verbindungen:
*  Standard SPI Port des Pro Mini zu LoRa ModuL
*     - Pro Mini-         -LoRa-Modul-
*     SS    (D10)          NSS
*     MOSI  (D11)          MOSI
*     MISO  (D12)          MISO
*     SCK   (D13)          SCK
*     Reset (RST)          RST
*     INT0  (D2)           DIO0         -- nicht genutzt
*/

#include <SPI.h>                                            // SPI Standard-Library, für die Kommunikation zwischen Prozessor und LoRa Modul
#include <Wire.h>                                           // I2C Standard-Library, hier nur für den I2C Test erforderlich   
#include <EEPROM.h>                                         // EEPROM Standard-Library zum Speichern von Daten
#include <LoRa.h>                                           // LoRa Library https://github.com/sandeepmistry/arduino-LoRa
#include <SSD1306AsciiWire.h>                               // SSD1306 Library mit Wire für die OLED   https://github.com/greiman/SSD1306Ascii

// #define Frequenz    434E6                                // (434E6) LoRa Arbeitsfrequenz festlegen, mit einem anderen Modul ist z.B. auch 868 MHz möglich
#define Frequenz 433175000                                  // selbst gewählte LoRa Arbeitsfrequenz
// #define Frequenz 433920000                               // andere wählbare LoRa Arbeitsfrequenz

#define I2C_ADDRESS 0x3C                                    // (0x3c) I2C Adresse definieren, siehe Datenblatt der OLED

SSD1306AsciiWire display;                                   // display Instanz erstellen

#define SS_PIN 10                                           // (NSS) LoRa-Modul Slave Select Pin
// #define DI0_PIN 2                                        // (DIO0) LoRa-Modul Digital Input/Output Pin 0, -- nicht benutzt

String receivedMessage;                                     // empfangene LoRa Nachricht
String sendMessage;                                         // zu sendene Nachricht

const int last_state  = 7;                                  // (7)  wenn an GND gelegt, wird der letzte Status des Relais gespeichert
const int Cod_0 = 8;                                        // (8)  Pin für HW Codierung bit "0"
const int Cod_1 = 9;                                        // (9)  Pin für HW Codierung bit "1"
const int Cod_2 = A0;                                       // (A0) Pin für HW Codierung bit "2"
const int Cod_4 = A1;                                       // (A1) Pin für HW Codierung bit "4"
const int Taste_Ein  = 3;                                   // (3)  Pin für Ein-Taste
const int Taste_Aus  = 4;                                   // (4)  Pin für Aus-Taste
const int green_LED  = 5;                                   // (5)  grüne LED (EIN)
const int blue_LED   = 6;                                   // (8)  blaue LED (AUS)

#define Title "LoRa-Schalter"                               // Überschrift in Display Beginnmeldung
#define Version "Version: 2.1"                              // Versionbezeichnung
#define Author "DL8LAB"                                     // Author 

String Ein_Stat;                                            // Quittungscode "EIN"
String Aus_Stat;                                            // Quittungscode "AUS"
String Ein_Code = "EIN";                                    // ("EIN") Einschaltcode
String Aus_Code = "AUS";                                    // ("AUS") Ausschaltcode
String Req_Rx = "RQ";                                       // ("RQ") Anfrage nach Status
String Req_Tx = "TQ";                                       // ("TQ") Anfrage nach Status

bool has_OLED;                                              // OLED ist angeschlossen
bool link_test;                                             // true, wenn Linkfehler Anzeige erforderlich
bool mem;                                                   // true, wenn der Schaltzustand gespeichert werden soll
unsigned long time_wait;                                    // Wartezeit bis zum Löschen der Meldung, wenn keine Anwort
unsigned long time_block;                                   // Zeit der Tastenblockade
const int time_blink = 160;                                 // (160) Zyklus einer LED-Blinkphase
const unsigned int c_wait = 2500;                           // (2500) Konstante für Quittungswartezeit
const unsigned int c_block  = 1500;                         // (1500) Konstante für Tastensperre
const unsigned int c_dspt = 300;                            // (300) Konstante für Anzeigedauer der Quittungsmeldung
byte rel_on;                                                // Status des Relais
byte HWcode;                                                // Variable für HW-Codierung


// ---------  Setup  ------------------------------------------------------------------------------
void setup() { 
 Ein_Stat = "Q"+ Ein_Code;                                  // Quittung muss sich vom EIN-Siganal unterscheiden
 Aus_Stat = "Q"+ Aus_Code;                                  // Quittung muss sich vom AUS-Siganal unterscheiden

 pinMode(Taste_Ein, INPUT_PULLUP);                          // Taste 1 als Input mit Pullupwiderstand konfigurieren
 pinMode(Taste_Aus, INPUT_PULLUP);                          // Taste 2 als Input mit Pullupwiderstand konfigurieren
 pinMode(Cod_0, INPUT_PULLUP);                              // Cod_0 als Input mit Pullupwiderstand konfigurieren
 pinMode(Cod_1, INPUT_PULLUP);                              // Cod_1 als Input mit Pullupwiderstand konfigurieren
 pinMode(Cod_2, INPUT_PULLUP);                              // Cod_2 als Input mit Pullupwiderstand konfigurieren
 pinMode(Cod_4, INPUT_PULLUP);                              // Cod_4 als Input mit Pullupwiderstand konfigurieren
 pinMode (last_state, INPUT_PULLUP);                        // last_state als Input mit Pullupwiderstand konfigurieren
 pinMode(green_LED, OUTPUT);                                // green_LED als Ausgang konfigurieren
 pinMode(blue_LED, OUTPUT);                                 // blue_LED als Ausgang konfigurieren

 blue_flash();                                              // Sub, erste Blinkmeldung zur Kontrolle, ob die blaue "Aus-LED" ok ist.

 Serial.begin(115200);                                      // seriellen Monitor mit 115200 baud starten
 while (!Serial)(blue_flash ());                            // warten bis serielle Schnittstell bereit ist, wenn nicht: Blinken (das kommt eigentlich nicht vor)

 Serial.print(F("\n\tLoRa Schalter mit Arduino Pro Mini 3,3V 8MHz -Author DL8LAB-")); // neue Zeile, Tab und Info ausgeben    
 Serial.print(F("\nSketch:   "));   Serial.println(__FILE__);                         // neue Zeile, Sketchnamen ausgeben
 Serial.print(F("geflasht am: "));   Serial.print(__DATE__);                          // Flashdatum ausgeben
 Serial.print(F(" um "));   Serial.print(__TIME__); Serial.println(F(" Uhr  \n"));    // Flashzeit ausgeben, neue Zeile
 delay(20);                                                                           // kurze Verzögerung

 cod_read();                                                                          // Sub, HW-Codierung lesen
 Serial.println (F(" -- Parameter --"));                                              // gestzte Parameter seriell ausgeben
 Serial.print (F("-Hardw.Codierung: ")); Serial.println (HWcode);                     // gelesenen HW-Codierung
 Serial.print (F("-Einschaltsignal: ")); Serial.println (Ein_Code);                   // gebildete Einschaltzeichenfolge
 Serial.print (F("-Ausschaltsignal: ")); Serial.println (Aus_Code);                   // gebildete Ausschaltzeichenfolge
 Serial.print (F("-Rx Status Abfr.: ")); Serial.println (Req_Rx);                     // gebildete Abfragezeichenfolge Rx
 Serial.print (F("-Tx Status Abfr.: ")); Serial.println (Req_Tx);                     // gebildete Abfragetzeichenfolge Tx
 Serial.print (F("-Status  -Ein-  : ")); Serial.println (Ein_Stat);                   // gebildete EIn-Status Zeichenfolge
 Serial.print (F("-Status  -Aus-  : ")); Serial.println (Aus_Stat);                   // gebildete Aus-Status Zeichenfolge
 Serial.print (F("-Tastensperre   : ")); Serial.print (c_block);                      // Zeit für gegenseitige Tastensperre
 Serial.print (F(" ms \n-Quittungszeit  : ")); Serial.print (c_wait); Serial.println (" ms");       // Zeit bis zum Anzeigen eines eventuelle Linkfehlers
 
 Serial.println (F(" -- Meldungen --"));                    // Ab hier werden Startmeldungen ausgegeben

 Wire.begin();                                              // starte two Wire Kommunikation 
 Serial.println(F("-Wire gestartet."));                     // Meldung seriell ausgeben  

 if (!LoRa.begin(Frequenz)) {                               // Frequenz ist zu am Anfang definiert worden (z.B. 434E6 Hz)
   Serial.println(F("-LoRa Initialisierungsfehler. Überprüfen Sie Ihre Verdrahtung!"));
   while (true)(blue_flash ());
  }
 Serial.println(F("-LoRa gestartet."));                     // LoRa Modul erkannt und gestartet  

// ---------  LoRa Parameter setzen ---------------------------------------------------------------
 LoRa.setCodingRate4(5);                                    // CodingRate 4/5 
 LoRa.setSignalBandwidth(125E3);                            // Bandwidth zb. 125kBaud - (125E3)
 // LoRa.setFrequency(Frequenz);                            // Frequenz kann auch hier gesetzt werden 
 // LoRa.enableCrc();                                       // enable CRC, CRC nicht aktiviert                  
 LoRa.setPreambleLength(8);                                 // (8) Preamble Länge
 LoRa.setSpreadingFactor(12);                               // (12) Spreading Faktor 7 - 12
 LoRa.setSyncWord(0x12);                                    // (0x12) Sync Word (0-0xFF)
 LoRa.setTxPower(14);                                       // (14) setze Power Level (2=min, 20=max)
 Serial.println(F("-LoRa Parameter gesetzt."));             // Meldung seriell ausgeben 
 Serial.print(F("-LoRa Frequenz: "));                       // gewählte 
 Serial.print(Frequenz);                                    // Freauenz
 Serial.println(F(" Hz"));                                  // in Hz seriell ausgeben 
//----------- LoRa Parameter setzen  ----- ENDE ------

//-----------  Schaltzustand abrufen   ------------------------------------------------------------ 
 if (digitalRead(last_state) == LOW)  {                     // (LOW) Lötbrücke gegen Masse ?
 Serial.println(F("-Schaltzustand wird gespeichert! ")) ; 
 mem = true;                                                // true, wenn der Schaltzustand gemerkt werden soll
 EEPROM.get(0, rel_on);                                     // den letzten Schaltzustand aus EEprom auslesen
 Serial.print("-letzter Schaltzustand: ") ;                 // Schaltzustand seriell ausgeben
 if (rel_on == 2) {Serial.println(" -Ein-");}               // 2 = Ein
 if (rel_on == 1) {Serial.println(" -Aus-");}               // 1 = Aus
  }
   else
  {
   Serial.println(F("-Schaltzustand wird -nicht- gespeichert."));
  }
   
//-----------  LED's schalten und Status senden   -----------------------------------------------
  if(rel_on==2) {                                           // Relais war zuletzt eingeschaltet
    digitalWrite (green_LED, HIGH);                         // Relais bzw. grüne LED einschalten
    digitalWrite(blue_LED, LOW);                            // blaue LED ausschalten 
    sendLoRaMessage(Ein_Stat);                              // "Ein" Status senden
    link_test=false;                                        // keine Linkfehler Anzeige erforderlich
   } 

  if(rel_on==1) {                                           // Relais war zuletzt ausgeschaltet
    digitalWrite (blue_LED, HIGH);                          // blaue LED einschalten 
    digitalWrite(green_LED, LOW);                           // grüne LED ausschalten 
    sendLoRaMessage(Aus_Stat);                              // "Aus" Status senden
    link_test=false;                                        // keine Linkfehler Anzeige erforderlich
   } 

 AdrTest_i2c();                                             // Sub prüfe, ob OLED angeschlossen ist

 if (has_OLED == true) {                                    // wenn OLED gefunden wurde
  
 display.begin(&Adafruit128x64, I2C_ADDRESS);               // OLED mite Wire Adresse starten
 display.setScrollMode(true);                               // Scrolling setzen
 // Hier einen Font setzen, sonst keine Anzeige auf dem OLED !!! 
 display.setFont(System5x7);                                // Standard-Font für Startmeldung setzen

//-----------  Startmeldung auf OLED --------------------------------------------------------------
  Serial.println(F("-OLED gestartet."));                    // Startmeldung seriell ausgeben
 	Serial.println(F("-zeige den OLED Startbildschirm."));
  display.clear();                                          // OLED löschen
  delay(20);                                                // kurze Verzögerung
  Meldg_Aus(Title,1,0);                                     // Überschrift mittig in Zeile 1 ausgeben
  Meldg_Aus(Version,3,0);                                   // Versionsnummer mittig in Zeile 3 ausgeben
  Meldg_Aus(Author,6,0);                                    // Author mittig in Zeile 6 ausgeben
  delay(2000);                                              // (2000) Anzeigedauer
  for (int i = 1; i <= 10; i++) {                           // nach oben schieben (scrollen)
    if (i == 10) {                                          // am Schluß 
      display.clear();                                      // OLED löschen
     }
    display.println();                                      // mit Zeilenvorschub nach oben schieben
    delay(80);                                              // (80) Scrollgeschwindigkeit
   }
  
 Meldg_Aus(Title,1,0);                                      // Überschrift auf OLED ausgeben
  
 display.setCursor(46,6);                                   // Cursor auf Pos 48, Zeile 6 
 display.print("Code " );                                   // "Code " ausgeben
 display.print(HWcode);                                     // HW Codierung (dezimal)  
   
 if (mem==true){                                            // wird der Schaltzustand gespeichert?
  display.setCursor(108,6);                                 // Coursor rechts unten
  display.print("M");                                       // Symbol für Speicherung schreiben.
   }  
 display.setFont(Arial_bold_14);                            // Font setzen für Status usw.
 } 
//-----------  Startmeldung Ende  -----------------------------------------------------------------
 
 blue_flash();                                              // Sub, Start-Blinkmeldung durch blaue "Aus-LED"
 Serial.println("-Los gehts...");                           // Startmeldung seriell ausgeben
  
 if (rel_on==0) {
    sendLoRaMessage(Req_Rx);                                // Status des Empfängers per LoRa abfragen
    Meldg_Aus("Link Test",3,2);                             // Meldung "Link Test" anzeigen
    link_test=true;                                         // Linkfehler Anzeige erforderlich
    }  
 if(rel_on==2)  {                                           // ist das Relais eingeschaltet?
    Meldg_Aus("- EIN -",3,3);                               // Meldung "- EIN -" anzeigen
    }
 if(rel_on==1) {
    Meldg_Aus("- AUS -",3,3);                               // Meldung "- AUS -"  anzeigen
    }    
    time_wait = millis() + c_wait;                          // Zeit + Quittungswartezeit merken
 }

//--------- Setup  ----- ENDE ------ 

//#################################################################################################
//###       Loop                                                                                ###
//#################################################################################################
void loop() {   

  if (digitalRead(Taste_Ein) == LOW && digitalRead(Taste_Aus) == HIGH && millis() >time_block){
  digitalWrite (green_LED, HIGH);                           // grüne LED einschalten
  digitalWrite (blue_LED, LOW);                             // blaue LED ausschalten
  EEPROM.put(0, 0);                                         // Status des Relais ("0") an Adresse 0 speichern
  sendLoRaMessage(Ein_Code);                                // "Ein" Befehl senden
  Meldg_Aus("Tx: Ein",3,2);                                 // zu sendende Meldung auf OLED und seriell ausgeben
  digitalWrite (green_LED, LOW);                            // grüne LED ausschallten  
  link_test = true;                                         // Linkfehler Anzeige erforderlich
  }
  
  if (digitalRead(Taste_Aus) == LOW && digitalRead(Taste_Ein) == HIGH && millis() >time_block){
  digitalWrite (blue_LED, HIGH);                            // blaue LED einschalten
  digitalWrite (green_LED, LOW);                            // grüne LED ausschalten 
  EEPROM.put(0, 0);                                         // Status des Relais ("0") an Adresse 0 speichern
  sendLoRaMessage(Aus_Code);                                // "Aus" Befehl senden
  Meldg_Aus("Tx: Aus",3,2);                                 // zu sendende Meldung auf OLED und seriell ausgeben
  digitalWrite (blue_LED, LOW);                             // blaue LED ausschalten
   link_test = true;                                        // Linkfehler Anzeige erforderlich
  }


// ---- Auf eingehende Nachrichten warten ---------------------------------------------------------
     if (LoRa.parsePacket()) {       
       receivedMessage = "";                                // String leeren           
       while (LoRa.available()) {                           // solange LoRa empfangene Zeichen vorhanden sind
       char mychar = (char)LoRa.read();                     // lese Buchstaben
       if (mychar>1){                                       // nur wenn char > 1
       receivedMessage += mychar;                           // String zusammensetzen
       }       
     }
// ---- Auf eingehende Nachrichten warten  ----- ENDE ------

//----- Rx Status senden  -------------------------------------------------------------------------
     if (receivedMessage==Req_Rx) {                         // wenn "Req_Rx" empfangen   
     Meldg_Aus ("Rx: Lora ok!",3,1);                        // Funkstrecke ok, auf OLED ausgeben, Text 1 seriell ausgeben
     Send_Status();                                         // Sub, Status senden
     }
//----- Rx Status senden  ----- ENDE ------

//----- Rx ok Relais aus  --------------------------------------------------------------------------
     if (receivedMessage==Req_Tx) {                         // wenn "Req_Tx" empfangen   nur Meldung
     Meldg_Aus ("Rx: LoRa ok!",3,1);                        // Funkstrecke ok, auf OLED ausgeben, Text 1 seriell ausgeben
     }
//----- Rx ok Relais aus  ----- ENDE ------


//----- Quittung empfangen?  ----------------------------------------------------------------------
      if (receivedMessage==Ein_Stat) {                      // Wenn "Quittung_EIN" empfangen   
      digitalWrite (green_LED, HIGH);                       // grüne LED einschallten
      digitalWrite (blue_LED, LOW);                         // blaue LED ausschalten
      Meldg_Aus("Status: -Ein-",3,1);                       // auf dem OLED in Zeile 3 schreiben, Text 1 seriell ausgeben
     }

      if (receivedMessage==Aus_Stat) {                      // Wenn "Quittung_AUS" empfangen   
      digitalWrite (green_LED, LOW);                        // grüne LED ausschallten
      digitalWrite (blue_LED, HIGH);                        // blaue LED einschalten
      Meldg_Aus("Status: -Aus-",3,1);                       // Auf dem OLED in Zeile 3 schreiben, Text 1 seriell ausgeben
     }   
//----- Quittung empfangen?  ----- ENDE ------

//----- Schaltsignal empfangen? -------------------------------------------------------------------

    if (receivedMessage==Ein_Code) { 
       digitalWrite (green_LED, HIGH);                      // grüne LED einschalten
       digitalWrite (blue_LED, LOW);                        // blaue LED auschalten
       EEPROM.put(0, 2);                                    // "2" in EEPROM speichern (Ein)
       sendLoRaMessage(Ein_Stat);                           // Quittung senden
       Meldg_Aus("Tx: Status Ein",3,2);                     // "Tx: Status Ein" auf OLDE Zeile 3 ausgeben, Text 2 seriell ausgeben
       rel_on=2;                                            // Relais Zustand "Ein" merken
       delay(c_dspt);                                       // Anzeigedauer der Meldung
       Meldg_Aus("- EIN -",3,1);                            // "- EIN -" auf OLDE Zeile 3 ausgeben, Text 1 seriell ausgeben
       link_test=false;                                     // keine Linkfehler Anzeige erforderlich
      }
    if (receivedMessage==Aus_Code) { 
       digitalWrite (green_LED, LOW);                       // grüne LED auschalten
       digitalWrite (blue_LED, HIGH);                       // blaue LED einschalten
       EEPROM.put(0, 1);                                    // "1" in EEPROM speichern (Aus)
       sendLoRaMessage(Aus_Stat);                           // Quittung senden
        Meldg_Aus("Tx: Status Aus",3,2);                    // "Tx: Status Aus" auf OLDE Zeile 3 ausgeben, Text 2 seriell ausgeben
       rel_on=1;                                            // Relais Zustand "Aus" merken; 200ms anzeigen
       delay(c_dspt);                                       // Anzeigedauer der Meldung
       Meldg_Aus("- AUS -",3,1);                            // "- AUS -" auf OLDE Zeile 3 ausgeben, Text 1 seriell ausgeben
       link_test=false;                                     // keine Linkfehler Anzeige erforderlich
      }    
//----- Schaltsignal empfangen?   ----- ENDE ------
  }
//----- Auf eingehende Nachrichten warten   ----- ENDE ------

//----- Linkfehler anzeigen  ----------------------------------------------------------------------
  if ((link_test == true) && (millis() > time_wait ))      // wurde eine Meldung gesendet, ist die Fehleranzeige erforderlich und die Zeit abgelaufen ?
   {
     Meldg_Aus("- kein Link -",3,4);               
     link_test=false;
   }
  //----- Linkfehler anzeigen   ----- ENDE ------
}
//#################################################################################################
//###     Ende Loop                                                                             ###
//#################################################################################################

//--------- Nachricht senden ----------------------------------------------------------------------
void sendLoRaMessage(String message) {
  LoRa.beginPacket();                                       // Aussendung vorbereiten
  LoRa.print(message);                                      // Nachricht senden
  LoRa.endPacket();                                         // Aussendung beenden
  // Serial.println("-LoRa gesendet:  " + message);         // zum Debuggen
  
  time_wait = millis() + c_wait;                            // Timer setzen für Quittungsempfang
  time_block = millis() + c_block;                          // Timer setzen für Tastenblockade
}
//---------- Nachricht senden   ----- ENDE ------


// ----- Funktion Meldungsausgabe -----------------------------------------------------------------
void Meldg_Aus(String DispText, int Zeile,int serMeld) {
// ---- gesendete Nachricht auf OLED anzeigen  ----------------------------------------------------
     unsigned int MsgLength = DispText.length();       
     display.setCursor(0,Zeile);                            // Cursor auf Pos 0, Zeile x
     display.print(F("                                          "));         // (52 Leerzeichen bei Arial14       
     display.setCursor((128-MsgLength*6)/2 ,Zeile);         // widh - Zeichen * 7 / 2 in Zeile x
     display.print(DispText);                               // Text auf OLED ausgeben

  switch (serMeld){                                         // Meldungen seriell ausgeben
      case 0:                                               // keine serielle Ausgabe erforderlich       
      break;                                                // switch() verlassen
      case 1:                                               // Meldung 1 "LoRa empfangen" seriell ausgeben
      Serial.print(F("-LoRa empfangen: ")); Serial.println(DispText);
      link_test= false;                                     // keine Linkfehler Anzeige erforderlich
      break;                                                // switch() verlassen
      case 2:                                               // Meldung 2 "LoRa gesendet" seriell ausgeben
      Serial.print(F("-LoRa gesendet:  "));Serial.println(DispText);
      break;                                                // switch() verlassen
      case 3:                                               // Meldung 3  "Relais-Status" seriell ausgeben
      Serial.print(F("-Relais-Staus:   "));Serial.println(DispText);
      break;                                                // switch() verlassen
      case 4:                                               // Meldung 4  "LoRa Linktest" seriell ausgeben
      Serial.print(F("-LoRa Linktest:  "));Serial.println(DispText);
      break;                                                // switch() verlassen
      default:                                              // sollte nicht vorkommen
      Serial.print(F("-nicht eindeutig!")); 
 }
}
// ----- Meldungsausgabe   ----- ENDE ------



// ----- Blinkmeldung blaue LED -------------------------------------------------------------------
void blue_flash () {
    digitalWrite (blue_LED,(!digitalRead (blue_LED)));      // Zustand der blauen LED wechseln
    delay(time_blink/2);                                    // halbe Blinkphase Pause
    digitalWrite (blue_LED,(!digitalRead (blue_LED)));      // blaue LED wieder auf ursprünglichen Zusatnd setzen
    delay(time_blink/2);                                    // halbe Blinkphase Pause
  }
// ----- Blinkmeldung blaue LED   ----- ENDE ------

// ----- Status senden   --------------------------------------------------------------------------
void Send_Status()
{
   if (rel_on==2)                                           // wenn "rel_on" = 2, dann
    {
     sendLoRaMessage(Ein_Stat);                             // Status "EIN" zurück senden
     Meldg_Aus ("-EIN-",3,2);                               // "-EIN-" auf OLED Zeile 3 ausgeben, Text 2 seriell ausgeben
    }
   if (rel_on==1)                                           // wenn "rel_on" = 1, dann
   {
      sendLoRaMessage(Aus_Stat);                            // Status "AUS" zurück senden
      Meldg_Aus ("-Aus-",3,2);                              // "-Aus-" auf OLED Zeile 3 ausgeben, Text 2 seriell ausgeben
   }
    if (rel_on==0)                                          // wenn "rel_on" = 0, dann
   {
      sendLoRaMessage(Req_Tx);                              // Status "Req Tx" zurück senden
      Meldg_Aus ("Tx: LoRa ok!",3,2);                       // "Tx: LoRa ok!" auf OLED Zeile 3 ausgeben, Text 2 seriell ausgeben
     
      digitalWrite (blue_LED,LOW);                          // Relais, bzw. blaue LED aus
      digitalWrite (green_LED,LOW);                         // grüne LED aus
    }
   }
// ----- Status senden    ----- ENDE ------


// ----- HW-Codierung lesen   ---------------------------------------------------------------------
void cod_read()
{
  if (digitalRead(Cod_0) == LOW) HWcode += 1;               // HWcode der Platine ermitteln und addieren (bit0)
  if (digitalRead(Cod_1) == LOW) HWcode += 2;               // HWcode der Platine ermitteln und addieren (bit1)
  if (digitalRead(Cod_2) == LOW) HWcode += 4;               // HWcode der Platine ermitteln und addieren (bit2)
  if (digitalRead(Cod_4) == LOW) HWcode += 8;               // HWcode der Platine ermitteln und addieren (bit3)

  //HWcode=12;                                              // zum Debuggen

  Ein_Code += HWcode;                                       // HW-Code an Einschalt-Code anhängen               
  Aus_Code += HWcode;                                       // HW-Code an Ausschalt-Code anhängen     
  Ein_Stat += HWcode;                                       // HW-Code an Einschalt-Quittung anhängen
  Aus_Stat += HWcode;                                       // HW-Code an Ausschalt-Quittung anhängen
  Req_Rx += HWcode;                                         // HW-Code an Rx-Quittungsabfrage anhängen
  Req_Tx += HWcode;                                         // HW-Code an Tx-Quittungsabfrage anhängen
}
// ----- HW-Codierung lesen    ----- ENDE ------


// ----- auf OLED prüfen   ------------------------------------------------------------------------
void AdrTest_i2c()
{
   Serial.println(F("-prüfe ob OLED vorhanden ist."));      // Meldung seriell ausgeben
   Wire.endTransmission(I2C_ADDRESS);                       // Wire mit OLED Adresse vorsorglich beenden                      
   Wire.beginTransmission(I2C_ADDRESS);                     // Starte "Wire" Kommunikation  
   if (Wire.endTransmission())                              // prüfe, ob OLED reagiert
   {                                                        // ergibt "1" wenn immer noch endTransmission Status gesetzt ist
    has_OLED=false;                                         // Flag auf false setzen
    Serial.println(F("-OLED nicht angeschlossen."));        
   }
   else                                                     // sonst "0"
   {
     has_OLED = true;                                       // Flag auf true setzen
     Serial.print(F("-OLED gefunden an Adresse: " ));       // Meldung seriell ausgeben
     Serial.print ("0x");                                   // "0x" voranstellen
     Serial.print (I2C_ADDRESS, HEX);                       // I2C Adresse seriell in Hex ausgeben
     Serial.print ("(Hex) ");                               // "Hex" anhängen
     Serial.print (I2C_ADDRESS, DEC);                       // I2C Adresse seriell in als Dezimalzahl ausgeben
     Serial.println ("(Dez)");                              // "Dez" anhängen
   }
}  
// ----- auf OLED prüfen   ----- ENDE ------
